#include <gst/gst.h>
#import "GStreamerBackend.h"
#import "VSLogging.h"

@implementation GStreamerBackend
{
    BOOL _hasStarted;
}

@synthesize
delegate = _delegate,
streamId = _streamId,
isActive = _isActive;

#pragma mark public interface methods

-(id) initWithDelegate:(id)delegate streamId:(NSString *)streamId
{
    TRC_ENTRY;
    if (self = [super init])
    {
        gst_debug_set_default_threshold(GST_LEVEL_INFO);
        
        _delegate = delegate;
        _streamId = streamId;
        
        _hasStarted = NO;
        _isActive = NO;
        _pipelineIsRunning = NO;
    }
    TRC_EXIT;
    return self;
}

- (void) start
{
    TRC_ENTRY;
    
    __weak GStreamerBackend *weakSelf = self;
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^
    {
        [weakSelf buildAndRunPipeline];
    });
    
    TRC_EXIT;
}

-(void) stop
{
    TRC_ENTRY;
    gst_element_send_event(_pipeline, gst_event_new_eos());
    TRC_EXIT;
}

/**
 * Utility method to get the current gstreamer version, so we can always be sure
 * what we're working with.
 */
-(NSString*) getGStreamerVersion
{
    char *version_utf8 = gst_version_string();
    NSString *version_string = [NSString stringWithUTF8String:version_utf8];
    g_free(version_utf8);
    return version_string;
}

#pragma mark utility methods

/**
 * Placeholder method for subclasses to define how they build their pipeline.
 */
-(NSString *) buildPipeline
{
    TRC_ENTRY;
    LOG_ERR(@"Child backend class must implement buildPipeline()");
    TRC_EXIT;
    return nil;
}

/**
 * Master method for building and running the pipeline.
 */
-(void) buildAndRunPipeline
{
    TRC_ENTRY;
    
    _hasStarted = NO;
    _pipelineIsRunning = NO;
    
    NSString *initialError = [self buildPipeline];
    
    //
    // wire up some listeners and try to start the pipeline
    
    if (initialError == nil)
    {
        if (gst_element_set_state(_pipeline, GST_STATE_PLAYING) == GST_STATE_CHANGE_FAILURE)
        {
            initialError = @"Unable to set pipeline in playing state";
            LOG_ERR(initialError, nil);
            gst_element_set_state(_pipeline, GST_STATE_NULL);
        }
    }
    
    if (initialError == nil)
    {
        [self pipelineStartRequestIssuedSuccessfully];
        [self setActive:YES];
    }
    
    if (initialError == nil)
    {
        [self runPipeline];
    }
    else
    {
        // alert the caller if we did not start successfully
        if (_delegate && [_delegate respondsToSelector:@selector(streamRequestFailed:reason:)])
        {
            [_delegate streamRequestFailed:self reason:initialError];
        }
        
        // cleanup our failed pipeline
        gst_element_set_state(_pipeline, GST_STATE_NULL);
        gst_object_unref (_pipeline);
        _pipeline = NULL;
    }
    
    TRC_EXIT;
}

/**
 * The running of the pipeline is broken out into a separate method so it can be executed
 * on any given queue.
 */
-(void) runPipeline
{
    //
    // start a main loop to keep the pipeline running
    // is any of this really needed since our application will continue to run without it?
    
    _pipelineIsRunning = YES;
    
    // create our own GLib main context and make it the default one
    _context = g_main_context_new ();
    g_main_context_push_thread_default(_context);
    
    // create and run a new loop
    _mainLoop = g_main_loop_new (_context, FALSE);
    
    // add a watch on the bus
    // this must happen after the context and main loop are created
    GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(_pipeline));
    gst_bus_add_watch(bus, gstcBusMessage, (__bridge gpointer)(self));
    g_object_unref(bus);
    
    LOG_DEBUG(@"Entering main loop");
    g_main_loop_run (_mainLoop);
    LOG_DEBUG(@"Exited main loop");
    
    // loop and context cleanup
    g_main_loop_unref (_mainLoop);
    g_main_context_pop_thread_default(_context);
    g_main_context_unref (_context);
    
    //
    // will fall-through when the loop is exited
    
    // make sure we deactivate when we leave the loop
    [self setActive:NO];
    _pipelineIsRunning = NO;
    
    gst_element_set_state(_pipeline, GST_STATE_NULL);
    gst_object_unref (_pipeline);
    _pipeline = NULL;
    
    [self pipelineDidStop];
}

-(void) pipelineDidStart
{
    TRC_ENTRY;
    TRC_EXIT;
}

-(void) pipelineDidStop
{
    TRC_ENTRY;
    TRC_EXIT;
}

// called when our pipeline is started for the first time
-(void) pipelineStartRequestIssuedSuccessfully
{
    TRC_ENTRY;
    TRC_EXIT;
}

/**
 * Utility method to check if a bus message source object is our pipeline.
 */
-(BOOL) objectIsPipeline:(GstObject *)object
{
    if (object == GST_OBJECT(_pipeline))
        return YES;
    
    return NO;
}

#pragma mark delegate utilities

/**
 * Post a message to the delegate
 */
-(void)setMessage:(NSString *) message
{
    if(_delegate && [_delegate respondsToSelector:@selector(streamMessage:message:)])
    {
        [_delegate streamMessage:self message:message];
    }
}

/**
 * update the active state, and trigger the active delegate if one has been defined
 */
-(void)setActive:(BOOL)isActive
{
    BOOL activeChanged = _isActive != isActive;
    _isActive = isActive;
    
    if (activeChanged)
    {
        NSString *activeDescription = isActive ? @"ACTIVE" : @"INACTIVE";
        LOG_DEBUG(@"(%@) changed to %@", _streamId, activeDescription);
        if (_delegate && [_delegate respondsToSelector:@selector(streamActiveChanged:isActive:)])
        {
            [_delegate streamActiveChanged:self isActive:_isActive];
        }
    }
}

#pragma mark gstreamer bus callbacks

-(void) gstLeaveMainLoop
{
    LOG_DEBUG(@"Leaving gst main loop");
    g_main_loop_quit(_mainLoop);
}

-(void) gstElementMessage:(GstBus *)bus message:(GstMessage *)msg
{
    TRC_ENTRY;
    TRC_EXIT;
}

-(void) gstPipelineStateChanged:(GstState)oldState newState:(GstState)newState pendingState:(GstState)pendingState
{
    if (newState == GST_STATE_PLAYING)
    {
        if (!_hasStarted)
        {
            _hasStarted = YES;
            [self pipelineDidStart];
        }
    }
}

/**
 * Pipeline element error callback
 */
static void gstcEos (GstBus *bus, GstMessage *msg, gpointer userData)
{
    GStreamerBackend *self = (__bridge GStreamerBackend *)userData;
    LOG_DEBUG(@"EOS received on bus");
    [self gstLeaveMainLoop];
}

/**
 * Pipeline element error callback
 */
static void gstcError (GstBus *bus, GstMessage *msg, gpointer userData)
{
    GStreamerBackend *self = (__bridge GStreamerBackend *)userData;
    
    GError *err = NULL;
    gchar *dbg;
    
    gst_message_parse_error(msg, &err, &dbg);
    gst_object_default_error(msg->src, err, dbg);
    
    gchar *errMsg = g_strdup_printf ("Error received from element %s: %s", GST_OBJECT_NAME (msg->src), err->message);
    [self setMessage:[NSString stringWithUTF8String:errMsg]];
    g_free(errMsg);
    
    g_error_free(err);
    g_free(dbg);
    
    [self gstLeaveMainLoop];
}

/**
 * Pipeline element state changed callback
 */
static void gstcElementStateChanged (GstBus *bus, GstMessage *msg, gpointer userData)
{
    GStreamerBackend *self = (__bridge GStreamerBackend *)userData;
    
    // only pay attention to messages coming from the pipeline, not its children
    if ([self objectIsPipeline:GST_MESSAGE_SRC (msg)])
    {
        GstState oldState, newState, pendingState;
        gst_message_parse_state_changed (msg, &oldState, &newState, &pendingState);
        
        gchar *message = g_strdup_printf("pipeline (%s) state changed to %s", [self.streamId cStringUsingEncoding:NSUTF8StringEncoding], gst_element_state_get_name(newState));
        LOG_DEBUG([NSString stringWithUTF8String:message], nil);
        g_free (message);
        
        [self gstPipelineStateChanged:oldState newState:newState pendingState:pendingState];
    }
}

/**
 * Pipeline element message callback
 */
static void gstcElementMesage(GstBus *bus, GstMessage *msg, gpointer userData)
{
    GStreamerBackend *self = (__bridge GStreamerBackend *)userData;
    [self gstElementMessage:bus message:msg];
}

/**
 * Primary bus message callback
 */
static gboolean gstcBusMessage (GstBus *bus, GstMessage *msg, gpointer userData)
{
    switch (GST_MESSAGE_TYPE(msg))
    {
        case GST_MESSAGE_EOS:
        {
            gstcEos(bus, msg, userData);
            break;
        }
            
        case GST_MESSAGE_ERROR:
        {
            gstcError(bus, msg, userData);
            break;
        }
            
        case GST_MESSAGE_STATE_CHANGED:
        {
            gstcElementStateChanged(bus, msg, userData);
            break;
        }
            
        case GST_MESSAGE_ELEMENT:
        {
            gstcElementMesage(bus, msg, userData);
            break;
        }
            
        default:
        {
            //LOG_TRACE(@"unhandled gst message type: %s", GST_MESSAGE_TYPE_NAME(msg));
            break;
        }
    }
    return TRUE;
}

@end

